<?php
// Include configuration
require 'config.php';

// Enable error logging (disable display in production)
ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

// Set headers
header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: https://tgbro.link'); // Restrict to specific origin
header('Access-Control-Allow-Methods: GET, POST');
header('Access-Control-Allow-Headers: Content-Type');

// Validate admin pin
$admin_pin = $_GET['admin_pin'] ?? $_POST['admin_pin'] ?? null;
if ($admin_pin !== $config['admin_pin']) {
    http_response_code(401);
    echo json_encode(['ok' => false, 'message' => 'Unauthorized access. Invalid or missing admin PIN.']);
    exit;
}

// Check PDO connection
if (!isset($pdo) || !($pdo instanceof PDO)) {
    error_log("Database connection failed: PDO not initialized", 3, 'error.log');
    http_response_code(500);
    echo json_encode(['ok' => false, 'message' => 'Database connection failed']);
    exit;
}

// Handle GET: Fetch all signals
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    try {
        $stmt = $pdo->query("SELECT signalId, status, name, price, shib_price, action, entry, tp, sl, pips, outcome, date FROM signals");
        $signals = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode(['ok' => true, 'signals' => $signals]);
    } catch (PDOException $e) {
        error_log("GET signal failed: " . $e->getMessage(), 3, 'error.log');
        http_response_code(500);
        echo json_encode(['ok' => false, 'message' => 'Database error', 'error' => $e->getMessage()]);
    }
    exit;
}

// Handle POST: Add, edit, delete
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['ok' => false, 'message' => 'Method not allowed']);
    exit;
}

$data = json_decode(file_get_contents("php://input"), true);
$action = $data['action'] ?? '';
$payload = $data['payload'] ?? [];

if (!in_array($action, ['add', 'edit', 'delete'])) {
    http_response_code(400);
    echo json_encode(['ok' => false, 'message' => 'Invalid action']);
    exit;
}

try {
    if ($action === 'add') {
        $signalId = bin2hex(random_bytes(8));
        $stmt = $pdo->prepare("INSERT INTO signals (signalId, status, name, price, shib_price, action, entry, tp, sl, pips, outcome, unlocked_users, date) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");
        $stmt->execute([
            $signalId,
            $payload['status'] ?? '',
            $payload['name'] ?? '',
            $payload['price'] ?? 0,
            $payload['shib_price'] ?? 0,
            $payload['action'] ?? '',
            $payload['entry'] ?? '',
            $payload['tp'] ?? '',
            $payload['sl'] ?? '',
            $payload['pips'] ?? '',
            $payload['outcome'] ?? '',
            '[]'
        ]);
        echo json_encode(['ok' => true, 'message' => 'Signal added', 'signalId' => $signalId]);
        exit;
    }

    if ($action === 'edit') {
        // Validate signalId
        if (empty($payload['signalId'])) {
            http_response_code(400);
            echo json_encode(['ok' => false, 'message' => 'Signal ID is required']);
            exit;
        }
        // Check if signal exists
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM signals WHERE signalId = ?");
        $stmt->execute([$payload['signalId']]);
        if ($stmt->fetchColumn() == 0) {
            http_response_code(404);
            echo json_encode(['ok' => false, 'message' => 'Signal not found']);
            exit;
        }
        // Build dynamic UPDATE query to preserve completed if not provided
        $fields = [];
        $params = [];
        $allowed_fields = ['status', 'name', 'price', 'shib_price', 'action', 'entry', 'tp', 'sl', 'pips', 'outcome'];

        foreach ($allowed_fields as $field) {
            if (isset($payload[$field])) {
                $fields[] = "`$field` = ?";
                $params[] = $payload[$field];
            }
        }
        if (empty($fields)) {
            http_response_code(400);
            echo json_encode(['ok' => false, 'message' => 'No fields provided to update']);
            exit;
        }
        $query = "UPDATE signals SET " . implode(', ', $fields) . ", date = NOW() WHERE signalId = ?";
        $params[] = $payload['signalId'];
        $stmt = $pdo->prepare($query);
        $stmt->execute($params);
        echo json_encode(['ok' => true, 'message' => 'Signal updated']);
        exit;
    }

    if ($action === 'delete') {
        if (empty($payload['signalId'])) {
            http_response_code(400);
            echo json_encode(['ok' => false, 'message' => 'Signal ID is required']);
            exit;
        }
        // Check if signal exists
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM signals WHERE signalId = ?");
        $stmt->execute([$payload['signalId']]);
        if ($stmt->fetchColumn() == 0) {
            http_response_code(404);
            echo json_encode(['ok' => false, 'message' => 'Signal not found']);
            exit;
        }
        $stmt = $pdo->prepare("DELETE FROM signals WHERE signalId = ?");
        $stmt->execute([$payload['signalId']]);
        echo json_encode(['ok' => true, 'message' => 'Signal deleted']);
        exit;
    }
} catch (Exception $e) {
    error_log("Action $action failed: " . $e->getMessage(), 3, 'error.log');
    http_response_code(500);
    echo json_encode(['ok' => false, 'message' => 'Database error', 'error' => $e->getMessage()]);
    exit;
}
?>