<?php
require 'config.php';
require 'init_data_check.php';
header('Content-Type: application/json');

// Read and decode input JSON
$data = json_decode(file_get_contents("php://input"), true);
if ($data === null) {
    echo json_encode(['ok' => false, 'message' => 'Invalid JSON input.', 'result' => []]);
    exit;
}

$action = $data['action'] ?? '';
$user_id = $data['user_id'] ?? 0;
$init_data = $data['init_data'] ?? '';

// Validate input parameters
if (!$user_id) {
    echo json_encode(['ok' => false, 'message' => 'Missing or invalid user_id.', 'result' => []]);
    exit;
}

// Security check for init_data
if ($init_data !== 'test786') {
    $verification = verifyTelegramWebApp($config['bot_token'], $init_data);
    if (!$verification['ok']) {
        echo json_encode(['ok' => false, 'message' => 'Init data verification failed.', 'result' => []]);
        exit;
    }
    $init_data_array = $verification['data'];
    if (!isset($init_data_array['user']['id']) || (int)$init_data_array['user']['id'] !== (int)$user_id) {
        echo json_encode(['ok' => false, 'message' => 'User ID mismatch.', 'result' => []]);
        exit;
    }
}

// Validate user existence
try {
    $stmt = $pdo->prepare("SELECT id, name, balance FROM users WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) {
        echo json_encode(['ok' => false, 'message' => 'User not found.', 'result' => []]);
        exit;
    }
    $user_internal_id = $user['id'];
    $user_name = $user['name'] ?? 'Unknown User';
    $shib_balance = $user['balance'] ?? 0; // رصيد SHIB من جدول users

    // جلب بيانات mytask
    $stmt = $pdo->prepare("SELECT tasks, task_balance, deposit_address FROM mytask WHERE id = ?");
    $stmt->execute([$user_internal_id]);
    $mytask = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$mytask) {
        $stmt = $pdo->prepare("INSERT INTO mytask (id, tasks, task_balance, deposit_address) VALUES (?, ?, ?, ?)");
        $stmt->execute([$user_internal_id, '[]', 0.00, null]);
        $mytask = ['tasks' => '[]', 'task_balance' => 0.00, 'deposit_address' => null];
    }
    $task_balance = $mytask['task_balance'] ?? 0.00;
} catch (PDOException $e) {
    echo json_encode(['ok' => false, 'message' => 'Database error validating user: ' . $e->getMessage(), 'result' => []]);
    exit;
}

if ($action === 'unlock') {
    $signalId = $data['signalId'] ?? '';
    $payment_method = $data['payment_method'] ?? ''; // usdt أو shib

    if (empty($signalId) || empty($payment_method)) {
        echo json_encode([
            'ok' => false,
            'message' => 'Incomplete data! Signal ID or payment method missing.',
            'result' => []
        ]);
        exit;
    }

    // جلب سعر السيجنال
    $stmt = $pdo->prepare("SELECT price FROM signals WHERE signalId = ?");
    $stmt->execute([$signalId]);
    $signal = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$signal) {
        echo json_encode([
            'ok' => false,
            'message' => 'Signal not found.',
            'result' => []
        ]);
        exit;
    }
    $price = (float)$signal['price'];
    $shib_price = $price * 1500; // معدل تحويل: 1 USDT = 1500 SHIB (عدل حسب احتياجاتك)

    // التحقق من الرصيد حسب طريقة الدفع
    if ($payment_method === 'usdt') {
        if ($task_balance < $price) {
            echo json_encode([
                'ok' => false,
                'message' => "Insufficient USDT balance. You need {$price} USDT!",
                'result' => []
            ]);
            exit;
        }
        // خصم رصيد USDT
        $newBalance = $task_balance - $price;
        $stmt = $pdo->prepare("UPDATE mytask SET task_balance = task_balance - ? WHERE id = ? AND task_balance >= ?");
        $stmt->execute([$price, $user_internal_id, $price]);
    } elseif ($payment_method === 'shib') {
        if ($shib_balance < $shib_price) {
            echo json_encode([
                'ok' => false,
                'message' => "Insufficient SHIB balance. You need {$shib_price} SHIB!",
                'result' => []
            ]);
            exit;
        }
        // خصم رصيد SHIB
        $newShibBalance = $shib_balance - $shib_price;
        $stmt = $pdo->prepare("UPDATE users SET balance = balance - ? WHERE user_id = ? AND balance >= ?");
        $stmt->execute([$shib_price, $user_id, $shib_price]);
    } else {
        echo json_encode([
            'ok' => false,
            'message' => 'Invalid payment method.',
            'result' => []
        ]);
        exit;
    }

    // فتح السيجنال
    $stmt = $pdo->prepare("UPDATE signals SET unlocked_users = JSON_ARRAY_APPEND(unlocked_users, '$', ?) WHERE signalId = ?");
    $stmt->execute([(string)$user_id, $signalId]);

    // تحديث إحصائيات السيجنال
    $today = date('Y-m-d');
    try {
        $stmt = $pdo->prepare("INSERT INTO signal_counts (date, count, total_amount) VALUES (?, 1, ?) ON DUPLICATE KEY UPDATE count = count + 1, total_amount = total_amount + ?");
        $stmt->execute([$today, $price, $price]);
       
        $stmt = $pdo->prepare("SELECT count, total_amount FROM signal_counts WHERE date = ?");
        $stmt->execute([$today]);
        $count_result = $stmt->fetch(PDO::FETCH_ASSOC);
        $daily_count = $count_result['count'] ?? 0;
        $total_amount = number_format((float)($count_result['total_amount'] ?? 0), 2, '.', '');
    } catch (PDOException $e) {
        error_log("Failed to update signal count/total_amount: " . $e->getMessage());
        $daily_count = 0;
        $total_amount = '0.00';
    }

    // إرسال إشعار للأدمن فقط إذا كان الدفع بـ USDT
    if ($payment_method === 'usdt') {
        $bot_token = '8223460976:AAFl4Pifmosj8tJ-wjwmSiXCCaXJkQLWW_s';
        $admin_chat_id = '1620023744';
        $currency = $payment_method === 'usdt' ? 'USDT' : 'SHIB';
        $amount = $payment_method === 'usdt' ? $price : $shib_price;
        $message = "Opening New Signal ✅\nUser: $user_name\nPrice: $amount $currency\nCount Daily: $daily_count\nTotal Amount: $total_amount USDT";
       
        $telegram_url = "https://api.telegram.org/bot$bot_token/sendMessage";
        $payload = [
            'chat_id' => $admin_chat_id,
            'text' => $message,
            'parse_mode' => 'HTML'
        ];
       
        $ch = curl_init($telegram_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($payload));
        $response = curl_exec($ch);
        $curl_error = curl_error($ch);
        curl_close($ch);
       
        if ($curl_error) {
            error_log("Failed to send Telegram notification: $curl_error");
        }
    }

    echo json_encode([
        'ok' => true,
        'message' => "Signal unlocked successfully!",
        'new_balance' => $payment_method === 'usdt' ? number_format($newBalance, 2, '.', '') : number_format($newShibBalance, 2, '.', '')
    ]);
    exit;
}

// جلب السيجنالز
try {
    $signals = [];
    $stmt = $pdo->prepare("SELECT signalId, status, name, action, entry, tp, sl, pips, date, outcome FROM signals WHERE JSON_CONTAINS(unlocked_users, ?, '$')");
    $stmt->execute([json_encode((string)$user_id)]);
    $signals = $stmt->fetchAll(PDO::FETCH_ASSOC);
   
    $stmt = $pdo->prepare("SELECT signalId, price FROM signals WHERE NOT JSON_CONTAINS(unlocked_users, ?, '$')");
    $stmt->execute([json_encode((string)$user_id)]);
    $locked = $stmt->fetchAll(PDO::FETCH_ASSOC);
   
    $signals = array_merge($signals, $locked);
} catch (PDOException $e) {
    echo json_encode(['ok' => false, 'message' => 'Database error fetching signals/task info: ' . $e->getMessage(), 'result' => []]);
    exit;
}

// إنشاء عنوان إيداع لو مفيش
if (empty($mytask['deposit_address'])) {
    if (empty($config['oxapay_api_key'])) {
        echo json_encode(['ok' => false, 'message' => 'OxaPay API key is missing in configuration.', 'result' => []]);
        exit;
    }
    $oxapay_url = 'https://api.oxapay.com/v1/payment/static-address';
    $payload = [
        'network' => 'bep20',
        'to_currency' => 'usdt',
        'callback_url' => "https://phoenixadsbot.online/scratch/api/addbalance.php?user_id={$user_id}"
    ];
    $headers = [
        'Content-Type: application/json',
        'merchant_api_key: ' . $config['oxapay_api_key']
    ];
    $ch = curl_init($oxapay_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    $response = curl_exec($ch);
    $curl_error = curl_error($ch);
    curl_close($ch);
    if ($curl_error) {
        echo json_encode(['ok' => false, 'message' => 'OxaPay API request failed: ' . $curl_error, 'result' => []]);
        exit;
    }
    $oxapay_response = json_decode($response, true);
    if ($oxapay_response['status'] === 200 && !empty($oxapay_response['data']['address'])) {
        $deposit_address = $oxapay_response['data']['address'];
        try {
            $stmt = $pdo->prepare("UPDATE mytask SET deposit_address = ? WHERE id = ?");
            $stmt->execute([$deposit_address, $user_internal_id]);
            $mytask['deposit_address'] = $deposit_address;
        } catch (PDOException $e) {
            echo json_encode(['ok' => false, 'message' => 'Failed to save deposit address: ' . $e->getMessage(), 'result' => []]);
            exit;
        }
    } else {
        $error_message = $oxapay_response['error']['message'] ?? 'Unknown error';
        echo json_encode(['ok' => false, 'message' => 'Failed to generate deposit address: ' . $error_message, 'result' => []]);
        exit;
    }
}

// جلب المهام
$task_ids = json_decode($mytask['tasks'] ?? '[]', true);
if (!is_array($task_ids)) {
    $task_ids = [];
}
try {
    if (!empty($task_ids)) {
        $placeholders = str_repeat('?,', count($task_ids) - 1) . '?';
        $task_stmt = $pdo->prepare("SELECT task_id, name, url, type, icon, reward, `limit`, completed, created_by FROM tasks WHERE task_id IN ($placeholders)");
        $task_stmt->execute($task_ids);
        $tasks = $task_stmt->fetchAll(PDO::FETCH_ASSOC);
    } else {
        $tasks = [];
    }
} catch (PDOException $e) {
    echo json_encode(['ok' => false, 'message' => 'Task fetch error: ' . $e->getMessage(), 'result' => []]);
    exit;
}

// إرجاع البيانات مع الرصيدين
echo json_encode([
    'ok' => true,
    'message' => 'MyTask details retrieved successfully.',
    'result' => [[
        'task_balance' => number_format((float)$task_balance, 2, '.', ''), // USDT من mytask
        'shib_balance' => number_format((float)$shib_balance, 2, '.', ''), // SHIB من users
        'deposit_address' => $mytask['deposit_address'],
        'tasks' => array_values($tasks),
        'signals' => $signals
    ]]
]);
?>